#!/usr/bin/env bash
# ------------------------------------------------------------------------------
# Enhanced Weather Fetcher with Postcode Validation
#
# Date: 2025-10-21 G0HLJ
#
# Description:
#   Fetches current weather data based on user-specified postcode and outputs
#   temperature in either Celsius or Fahrenheit as per the INI configuration file.
#   If coordinates are not provided, it fetches them via an API and checks if they are valid.
# ------------------------------------------------------------------------------
set -euo pipefail
export PATH="/usr/local/sbin:/usr/local/bin:/usr/sbin:/usr/bin:/sbin:/bin:$PATH"

# ---------- Configuration ----------
# INI file for configuration
config_file="weather_config.ini"

# Read the INI file values for [weather] section
while IFS='=' read -r key value; do
  # Remove leading/trailing whitespace
  key=$(echo "$key" | xargs)
  value=$(echo "$value" | xargs)
  
  # Skip empty lines and lines starting with '#'
  if [[ -n "$key" && "$key" != \#* && "$key" != [coordinates] ]]; then
    # Set the weather-related variables directly
    if [[ "$key" == "postcode" ]]; then
      postcode="$value"
    elif [[ "$key" == "temperature_unit" ]]; then
      temperature_unit="$value"
    fi
  fi
done < <(grep -v '^\[.*\]' "$config_file")

# Check if the postcode is empty and print "Invalid postcode" if so
if [[ -z "$postcode" ]]; then
  echo "Invalid postcode"
  exit 1
fi

# Default to DN5 8QW if no postcode is provided (though we've added validation)
postcode=${postcode:-"DN5 8QW"}
temperature_unit=${temperature_unit:-"C"}

# ---------- API to Fetch Coordinates ----------
# Function to get latitude and longitude based on postcode
get_coordinates_from_postcode(){
  local postcode="$1"
  
  # Call the Postcode API (example API: "https://api.postcodes.io/postcodes/{postcode}")
  response=$(curl -fsS "https://api.postcodes.io/postcodes/${postcode}" || true)
  
  if [[ -z "$response" || "$(echo "$response" | jq -r '.status')" != "200" ]]; then
    echo "Error: Unable to fetch coordinates for postcode $postcode."
    return 1
  fi

  # Extract latitude and longitude from the response
  lat=$(echo "$response" | jq -r '.result.latitude')
  lon=$(echo "$response" | jq -r '.result.longitude')

  # Check if latitude and longitude are valid
  if [[ "$lat" == "null" || "$lon" == "null" ]]; then
    echo "Invalid postcode: Coordinates could not be fetched."
    return 1
  fi

  echo "$lat,$lon"
}

# ---------- Read or Fetch Coordinates ----------
# Check if coordinates exist in the array or fetch them
declare -A postcode_coords
while IFS='=' read -r pcode coords; do
  # Remove leading/trailing whitespace
  pcode=$(echo "$pcode" | xargs)
  coords=$(echo "$coords" | xargs)
  
  # Skip empty lines and lines starting with '#'
  if [[ -n "$pcode" && "$pcode" != \#* && "$pcode" != [weather] ]]; then
    postcode_coords["$pcode"]="$coords"
  fi
done < <(grep '^[^#]' "$config_file" | sed -n '/^\[coordinates\]/,/^\[/{/^[^#]/p}')

# Check if postcode exists in the array, otherwise fetch the coordinates from API
if [ -z "${postcode_coords[$postcode]+x}" ]; then
  echo "Postcode Error '$postcode' Check the config File"
  
  # Fetch coordinates using the API
  coords=$(get_coordinates_from_postcode "$postcode")
  
  # If fetching the coordinates failed, exit
  if [ $? -ne 0 ]; then
    echo "Invalid postcode"
    exit 1
  fi

  postcode_coords["$postcode"]="$coords"
  
  # Optionally, add the new coordinates to the config file (if you want to save them)
  echo "Adding $postcode with coordinates $coords to the config file..."
  sed -i "/\[coordinates\]/a $postcode = $coords" "$config_file"
fi

# Extract latitude and longitude from the postcode
coords="${postcode_coords[$postcode]}"
lat=$(echo "$coords" | cut -d',' -f1)
lon=$(echo "$coords" | cut -d',' -f2)

# ---------- Helpers ----------
# Convert Fahrenheit to Celsius
convert_to_celsius(){
  local temp_f="$1"
  echo "$(awk -v f="$temp_f" 'BEGIN{printf "%.1f", (f-32)*5/9}')"
}

# Convert Celsius to Fahrenheit
convert_to_fahrenheit(){
  local temp_c="$1"
  echo "$(awk -v c="$temp_c" 'BEGIN{printf "%.1f", (c*9/5)+32}')"
}

# Fetch weather data from Open-Meteo
fetch_weather(){
  local data temp_f temp_c cond is_day

  # Fetch weather data (temperature and condition)
  data=$(curl -fsS "https://api.open-meteo.com/v1/forecast?latitude=${lat}&longitude=${lon}&current=temperature_2m,weather_code,is_day&temperature_unit=fahrenheit&timezone=auto" || true)

  if [ -z "$data" ]; then
    echo "Error: No weather data available"
    return 1
  fi

  # Extract temperature in Fahrenheit, condition, and day/night info
  temp_f=$(echo "$data" | sed -n 's/.*"temperature_2m":\s*\([-0-9.]\+\).*/\1/p' | head -n1)
  cond=$(echo "$data" | sed -n 's/.*"weather_code":\s*\([0-9]\+\).*/\1/p' | head -n1)
  is_day=$(echo "$data" | sed -n 's/.*"is_day":\s*\([0-9]\).*/\1/p' | head -n1)

  if [ -z "$temp_f" ]; then
    echo "Error: No temperature data available"
    return 1
  fi

  # Convert to Celsius
  temp_c=$(convert_to_celsius "$temp_f")

  # Map weather codes to conditions
  case "$cond" in
    0) cond="clear" ;;
    1|2) cond="sunny" ;;
    3) cond="cloudy" ;;
    45|48) cond="fog" ;;
    51|53|55|56|57) cond="rain" ;;
    61|63|65|66|67|80|81|82) cond="rain" ;;
    71|73|75|77|85|86) cond="snow" ;;
    95|96|99) cond="thunderstorm" ;;
    *) cond="clear" ;;
  esac

  # If it's night, modify the condition (e.g., "sunny" becomes "clear night" instead)
  if [ "$is_day" -eq 0 ]; then
    if [[ "$cond" == "sunny" ]]; then
      cond="clear night"
    else
      cond="${cond} night"
    fi
  fi

  # Output the results based on the temperature unit
  if [ "$temperature_unit" == "F" ]; then
    temp_out="$temp_f°F"
  else
    temp_out="$temp_c°C"
  fi

  echo "$temp_out|$cond"
}

# ---------- Run ----------
result=""
result=$(fetch_weather)

if [ -z "$result" ]; then
  echo "No weather data available"
  exit 1
fi

# Split the result into temperature and condition
temp_out="${result%%|*}"
cond="${result##*|}"

# Save temperature and condition to output files
echo "$temp_out" > "/tmp/temperature"
echo "$cond" > "/tmp/condition.gsm"

# Print the result to the console
echo "$temp_out, Condition: $cond"

exit 0
